/*
 * Program: JetiLog
 * Version: 3.0
 * Date: 160104
 * Author : Jennifer Sirtl
 * HW-Platform: Arduino Micro
 *
 * Purpose of Program:
 *
 * Log alle on the Jetibox displayed informations into a SD-Card in "Samplingrate_ms" grid. Each Power up generates a new Flight-File. Cutting Power (switch of Transmitter
 * causes a regular File close (safe) if my published schematic is used.
 *
 * Versions:
 *
 * 0.1 First Version, olny Serial output, straight forward parser
 * 1.1 Version with SD-Card Log, first measurements, no timer-gating
 * 2.0 New Structure, msg-System, better parser, safe shutdown-function
 * 2.1 Function hub: Selectable Logger behavior, idle-couter, enhanced logtimer
 * 3.0 Inclusion of an RTC
 *
 * Known Restrictions:
 *
 * Time Limitation per File
 *   Log per File has to be shorter than 49,3Days (Overrun of TSLog). No damage if occurs, but for one period of 
 *   Samplingrate_ms, all changes are logged independent of Samplingrate_us.
 *
 * The MIT License (MIT)
 * Copyright (c) 2015 Jennifer Sirtl
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
 
//Log-Control
#define LnC  7 //Log all received Messages / Log if changes detected
#define LnT  6 //Log with no Time-Limitation /Log in time-gid, limiting Filesize
#define LaM  5 //Log all Messages /Log only Messages, containing numbers only
#define Prs  4 //Parse / direct Log 
#define CtS  3 //comma to Semicolon conversion (needs PRS=1)
#define LcE  1 //Log content Error /Log no Errors
#define LcK  0 //Log content Keystroke/ Log no Keystrokes
#define RV3029Write 0xac/2
#define CtrlADR 0x00
#define ClockADR 0x08
#define EEPRCTRL 0x30
#define STR2BCD_ERROR 0xff


//Libs needed
#include <stdlib.h>
#include <avr/interrupt.h>
#include <Arduino.h>
#include <SdFat.h>
#include <Jetilog2_x.h>
#include <Wire.h>

//Pinnings
const uint8_t chipSelect = SS;
const uint8_t SDFlushAnalog = 7; //Pin PE6/D7/A0

//Logger Settings
const uint32_t Samplingrate_ms =500; //Samplint rate in ms
#define FILE_BASE_NAME "" //Filename-Prefix

//Jeti Settings
#define l_UBRR1  (((unsigned long) F_CPU / 16 / 9600) - 1)//Calculate Diver, Mode "Asynch.Normal" (U2XN=0)

//Objects and Variables
SdFat sd;// File system object
SdFile file;// Log file
uint32_t TSLog;//Time in ms for next data record
uint32_t TSStart;//Time in ms Start logging
uint8_t SDCloseFile = 0;//Indicator that SDFile has to be closes imedeately 0=no, 1=yes,close it
char fileName[13] = FILE_BASE_NAME "yymmdd00.CSV";
const uint8_t BASE_NAME_SIZE = sizeof(fileName) - 7;
uint32_t idleCnt=0;
uint8_t LogCtrl=0; //Log-Control defines behavior of Logger
uint8_t debug;

//Global Variables
msgStruct msg[countOfmsg+1];//Memory for Messages;
msgStruct *inPTR;
msgStruct *oldPTR;
msgStruct *filledPTR;//pointer to Message
msgStruct *emptyPTR;
uint8_t CsrPos;

CSVSetStruct CSVSet;
CSVSetStruct *CSVSetPTR;

uint16_t msgcounter;

//Error hanling
#define error(msg) error_P(PSTR(msg))// Error messages stored in flash
//-------------------------------------
void error_P(const char* msg) {
  sd.errorHalt_P(msg);
}
//-------------------------------------

msgStruct * FindEmptyMsg(){
  uint8_t i;
  msgStruct *	EmptymsgPTR = NULL;
  msgStruct *	ForcemsgPTR = NULL;
  uint8_t tstampMin;
  msgStruct * ptr;
  
  cli();
  ptr=NULL;
  if (emptyPTR !=NULL) {//Fast mode: emptyMsg always defined (prio1)
    ptr=emptyPTR;
    emptyPTR=NULL;
  }
  else{//search emptyMsg
    i=0;
    while ((EmptymsgPTR==NULL)&&(i<=countOfmsg)){//in all messages
      if (msg[i].msgstatus==0) EmptymsgPTR= &msg[i];//if message has statuf free, take that (prio2)
      else{
        if ((msg[i].msgstatus==2)&&((msg[i].tstamp<=tstampMin)||(ForcemsgPTR==NULL))){//otherwise find oldest message wich is filled and skip them (prio3)
          ForcemsgPTR= &msg[i];
          tstampMin= msg[i].tstamp;
        }
      }
      i++;
    }
    if (EmptymsgPTR!=NULL) {
      ptr=EmptymsgPTR;
      Serial.println("found");

    }
    else{
      if (ForcemsgPTR!=NULL) {
        ptr=ForcemsgPTR;
        ptr->msgstatus=0;
        Serial.println("force");
      }
      else{
          for (i = 1;i<=countOfmsg;i++){  
            msg[i].text[0]=0x00;
            msg[i].ID=i;
            msg[i].msgstatus=0;
          }
          emptyPTR = NULL;
          oldPTR = NULL;
          inPTR = NULL;
          filledPTR = NULL;
          ptr=&msg[0]; //if there was not possible to create a emptyMsg, initate new (prio4)
          Serial.println("stall");
      } 
    }
  }
  if (ptr!=NULL) ptr->tstamp=millis();
  sei();
  return(ptr);
}


void Msg2CSV(msgStruct *msg, CSVSetStruct *CSVSet){
  uint8_t pIsNum,aIsNum;
  uint8_t source,destination;//Cursors for parser
  char achar;
  destination=0;
  source=0;
  achar=msg->text[source];
  uint8_t hasNumbers=0;

  while(achar!=0x00){ //Parse all Characters
    //Detect Numbers----------------------------
    if (strchr("0123456789", achar)){
      // Number detected
      if (!hasNumbers) hasNumbers=1; 
      aIsNum=1; 
    }
    else{
      //no Number detected
      aIsNum=0;
    }///---------------------------------------       
    switch (achar){     
    case 0x2E: //Handling of commas an ...
      if (LogCtrl&(1<<CtS)) achar=44;
      if (pIsNum) {
        aIsNum=1;
      }
      else {
        if (strchr("0123456789", msg->text[source+1])) aIsNum=1;
      }
      break;
    case '-': //Handling of minus and --- 
      if (strchr("0123456789", msg->text[source+1])) aIsNum=1;
      break;

    case '+': //Handling of minus and --- 
      if (strchr("0123456789", msg->text[source+1])) aIsNum=1;
      break;
    case 0xDF:
      achar='°';
      break;  
    default:
      //normal character
      break;
    }				  
    if ((pIsNum!=aIsNum)&&(source!=0)&&(LogCtrl&(1<<Prs))){//transient between Numbers and Character
      CSVSet->text[destination]=';';
      destination++;
    } 
    CSVSet->text[destination]=achar;
    destination++;
    source++;
    achar=msg->text[source];
    pIsNum=aIsNum;
  }//End of Parse all chars			
  CSVSet->text[destination]=0x00; //Terminate CSVSet
  CSVSet->hasNumbers=hasNumbers;
}

ISR(ANALOG_COMP_vect){
  SDCloseFile=1; //close File as fast as possible
}

ISR(USART1_RX_vect){
  uint8_t achar; //Character has been received
  uint8_t RXStatus;//Status of received Data
  uint8_t RXContent_NoFrame;//Flag, if received char is content (=1) or frame (=0)

  cli(); 
  RXStatus= UCSR1A;
  RXContent_NoFrame=UCSR1B;
  achar=UDR1;
  RXContent_NoFrame=(RXContent_NoFrame>>1) & 0x01;
  if ((RXStatus & ((1<<FE1)|(1<<DOR1)|(1<<UPE1)))){
    //Error in RX
    if (LogCtrl&(1<<LcE)) {
      if (inPTR==NULL) inPTR=FindEmptyMsg();
     }
    if (RXStatus & (1<<FE1)){
      delay(10);
      if (LogCtrl&(1<<LcE)){
        if (inPTR==NULL) inPTR=FindEmptyMsg();
        if (inPTR!=NULL){
          inPTR->msgstatus=1;//0=empty, 1=RxReserved, 2=filled, 3=LogReserved
          strcpy (inPTR->text,"Error: Frame ");
          inPTR->msgstatus=2;//0=empty, 1=RxReserved, 2=filled, 3=LogReserved
          filledPTR=inPTR;
          inPTR=NULL;               
          Serial.print("Error: Frame ");
        } 
      }
    }  
  
   
    if (RXStatus & (1<<DOR1)){
      delay(10);
      if (LogCtrl&(1<<LcE)){
        if (inPTR==NULL) inPTR=FindEmptyMsg();
        if (inPTR!=NULL){
          inPTR->msgstatus=1;//0=empty, 1=RxReserved, 2=filled, 3=LogReserved
          strcpy (inPTR->text,"Error: Overrun ");
          inPTR->msgstatus=2;//0=empty, 1=RxReserved, 2=filled, 3=LogReserved
          filledPTR=inPTR;
          inPTR=NULL;               
          Serial.print("Error: Overrun ");
        }  
      }

    }    
    if (RXStatus & (1<<UPE1)){
      delay(10);
      if (LogCtrl&(1<<LcE)){
        if (inPTR==NULL) inPTR=FindEmptyMsg();
        if (inPTR!=NULL){
          inPTR->msgstatus=1;//0=empty, 1=RxReserved, 2=filled, 3=LogReserved
          strcpy (inPTR->text,"Error: Parity ");
          inPTR->msgstatus=2;//0=empty, 1=RxReserved, 2=filled, 3=LogReserved
          filledPTR=inPTR;
          inPTR=NULL;          
          Serial.print("Error: Parity ");
        }
      }

    }
  }
  else {//No Error
    if (!RXContent_NoFrame){ //Frame-Handling
      switch (achar){
      case 0xF0:
        break;
        
      case 0x70:
        if (LogCtrl&(1<<LcK)) {
          if (inPTR==NULL) inPTR=FindEmptyMsg();
          if (inPTR!=NULL){
            inPTR->msgstatus=1;//0=empty, 1=RxReserved, 2=filled, 3=LogReserved
            strcpy (inPTR->text,"->CMD Button Left");
            inPTR->msgstatus=2;//0=empty, 1=RxReserved, 2=filled, 3=LogReserved
            filledPTR=inPTR;
            inPTR=NULL;            
          }
          Serial.println("->CMD Button Left");
        }  
        break;
        
      case 0xB0:
        if (LogCtrl&(1<<LcK)) {
          if (inPTR==NULL) inPTR=FindEmptyMsg();
          if (inPTR!=NULL){
            inPTR->msgstatus=1;//0=empty, 1=RxReserved, 2=filled, 3=LogReserved
            strcpy (inPTR->text,"->CMD Button Down");
            inPTR->msgstatus=2;//0=empty, 1=RxReserved, 2=filled, 3=LogReserved
            filledPTR=inPTR;
            inPTR=NULL;            
            Serial.println("->CMD Button Down");
          }
          else{
            Serial.println("Error->CMD Button Down"); 
          }
        }  
        break;
        
      case 0xD0:
        if (LogCtrl&(1<<LcK)) {
          if (inPTR==NULL) inPTR=FindEmptyMsg();
          if (inPTR!=NULL){
            inPTR->msgstatus=1;//0=empty, 1=RxReserved, 2=filled, 3=LogReserved
            strcpy (inPTR->text,"->CMD Button UP");
            inPTR->msgstatus=2;//0=empty, 1=RxReserved, 2=filled, 3=LogReserved
            filledPTR=inPTR;
            inPTR=NULL;

          }
          Serial.println("->CMD Button UP");
        }  
        break;
        
      case 0xE0:
        if (LogCtrl&(1<<LcK)) {
          if (inPTR==NULL) inPTR=FindEmptyMsg();
          if (inPTR!=NULL){
            inPTR->msgstatus=1;//0=empty, 1=RxReserved, 2=filled, 3=LogReserved
            strcpy (inPTR->text,"->CMD Button Right");
            inPTR->msgstatus=2;//0=empty, 1=RxReserved, 2=filled, 3=LogReserved
            filledPTR=inPTR;
            inPTR=NULL;
          }
          Serial.print("->CMD Button Right");   
          Serial.print("/");
          Serial.print(msg[0].msgstatus);
          Serial.print("/");
          Serial.print(msg[1].msgstatus);
          Serial.print("/");
          Serial.println(msg[2].msgstatus);       
        }  
        break;
        
      case 0xFE: //New message
        if (inPTR==NULL) inPTR=FindEmptyMsg();
        if (inPTR!=NULL){
          inPTR->msgstatus=1;//0=empty, 1=RxReserved, 2=filled, 3=LogReserved
          CsrPos=0;
          inPTR->text[0]=0x00;
        }
        else{
          Serial.println("Err:no empty msg");
        }	
        break;  
        
      case 0xFF: //Message has been finished
         if (inPTR!=NULL){
          inPTR->text[CsrPos]=0x00;
          inPTR->msgstatus=2;//0=empty, 1=RxReserved, 2=filled, 3=LogReserved
          filledPTR=inPTR;
          inPTR=NULL;
        }
        break;
        
      default: 
        //All other frames
        break;	
      }//end of switch
    }//end of if/Frame
    else{ //Normal Character
      if (inPTR!=NULL){
        if ((achar>31)&&(achar<254)){
          inPTR->text[CsrPos]=achar;
          if (CsrPos<msg_length_max) CsrPos++;
        }
      }	
    }//end normal Character
  }//end of no Error
  sei();
}

void PowerDown (){
  file.close();  
  Serial.println(F("Done"));
  while(1) {
  }//Kill Job 
}

void EnableIRQRx1(){
  // Set baudrate for Jeti
  UCSR1A &= ~(1<<U2X1);//U2X1 disabled, Divisor of BdRate is 16
  UBRR1H= uint8_t(l_UBRR1>>8);//high byte
  UBRR1L= uint8_t(l_UBRR1 & 0xFF);//low byte

  // Set frame format: Asynchronous, odd parity, 2 stop bits, 9 data bits
  UCSR1C = (1<<UPM11)|(1<<UPM10)|(1<<USBS1)|(1<<UCSZ10)|(1<<UCSZ11);
  UCSR1B = (1<<UCSZ12);

  // Enable receiver , enable RX-interrupt
  UCSR1B |= (1<<RXCIE1)|(1<<RXEN1);  
}

void EnableIRQAnalogComp(){
  pinMode(SDFlushAnalog, INPUT);
  //DIDR1 |= (1<<AIN0D);
  ACSR  |=  (0<<ACI);      // clear Analog Comparator interrupt
  ADCSRB |= (0<<ACME);     // Connect Bandgap to AIN- Input of Comparator
  ACSR  |=
    (0<<ACD)   |         // Comparator ON
    (0<<ACBG)  |         // Connect AIN0/PE6 to AIN+ Input of Comparator
    (0<<ACO)   |         // Analog Comparator Output
    (0<<ACIC)  |         // input capture disabled
    (1<<ACIS1) |         // set interrupt bit on falling edge
    (0<<ACIS0);          //    (ACIS1 and ACIS0 == 11)
  // Enable the interrupt
  ACSR |= (1 << ACIE);
}

void SetupMSG(){
  uint8_t i;
  //Init Msg-System
  msgcounter=0;
  msg[0].text[0]=0x00; //Message
  //msg[0].text[38]=0x00; //Message
  msg[0].msgstatus=0;
  msg[0].tstamp=millis();
  msg[0].ID=0;
  
  for (i = 1;i<=countOfmsg;i++){  
    msg[i]=msg[0];
    msg[i].ID=i;
    msg[i].msgstatus=0;
  }
  emptyPTR = &msg[0];
  oldPTR = &msg[1];
  inPTR = NULL;
  filledPTR = NULL;
}

void SetupFilename(){
  byte data[8];
  byte index=0;
  char buffer[4];

  
  Wire.beginTransmission(RV3029Write);
  Wire.write(ClockADR);
  Wire.endTransmission();
  
  Wire.requestFrom(RV3029Write, 8);  
  while(Wire.available()) 
  { 
    data[index] = Wire.read(); 
    index++;
  }
  sprintf(buffer,"%02x",data[6]); //jahr
  fileName[0]=buffer[0];
  fileName[1]=buffer[1];
  
  sprintf(buffer,"%02x",data[5]); //monat
  fileName[2]=buffer[0];
  fileName[3]=buffer[1];
  
  sprintf(buffer,"%02x",data[3]); //tag  
  fileName[4]=buffer[0];
  fileName[5]=buffer[1];
  
  // Find an unused file name.
  /*
  if (BASE_NAME_SIZE > 6) {
    error("FILE_BASE_NAME too long");
  }
  */
  while (sd.exists(fileName)) {
    if (fileName[BASE_NAME_SIZE + 1] != '9') {
      fileName[BASE_NAME_SIZE + 1]++;
    } 
    else if (fileName[BASE_NAME_SIZE] != '9') {
      fileName[BASE_NAME_SIZE + 1] = '0';
      fileName[BASE_NAME_SIZE]++;
    } 
    else {
      error("Can't create file name");
    }
  }
}  



uint8_t HasDifferences(msgStruct * msgPtr,msgStruct * msg2Ptr){
  uint8_t msgZ;
  char achar,achar2;
  uint8_t difference;
  uint8_t index,index2;
  
  index=0; 
  index2=0;
  difference=0;
  msgZ=0;

  if (msgPtr!=NULL) {
    index=msgPtr->ID; 
    if (msg2Ptr!=NULL){
      index2=msg2Ptr->ID; 
      achar=msg[index].text[msgZ];
      achar2=msg[index2].text[msgZ];
 
      do{
        if (achar!=achar2) difference=msgZ;
        msgZ++;
        achar=msg[index].text[msgZ];
        achar2=msg[index2].text[msgZ];
      } while((achar!=0x00)&&(achar2!=0x00)&& ((difference==0)||(msgZ==1)));//Parse all Characters, first Char is not observed
      if ((difference==0) && (achar!=achar2)) difference=msgZ;
    }
  }
  else{
    difference=2;//if one msg is NULL
  }  
  
  return(difference);
}


void setup() {

  LogCtrl = ( //LnC,LnT,LaM,LcE,LcK
    (0<<LnC) |
    (0<<LnT) |
    (0<<LaM) |
    (1<<Prs) |
    (1<<CtS) |
    (0<<LcE) |
    (1<<LcK));

  Wire.begin();
  Wire.beginTransmission(RV3029Write);
  Wire.write(CtrlADR);
  Wire.write(B11101001); //Control_1:
                         //CLKout -> CLKout-Pin
                         //Countdown Clock <= 0,5s
                         //enable Selfrecovery
                         //disable EEPROM refresh each hour
                         //disable Countdown
                         //disable Timer
                         //enable  1Hz Clock Source for watch           
  //          76543210  Bitschablone
  Wire.write(B00010100); //Control_INT: prüfen
  Wire.write(B00000000); //Control_INT Flag: disable all Interrupts and Alarms, clear PON
  Wire.write(B00000000); //Control_Status: disable all Interrupts and Alarms
  //Wire.write(B00000000); //Control_Reset: No Reset
  Wire.endTransmission(); 


  Wire.beginTransmission(RV3029Write);
  Wire.write(EEPRCTRL);
  //Wire.write(B00000001); //EEPROM_Control //Reset Value 
  Wire.write(B00001111); //EEPROM_Control: 
                         //Disable Trickle-Charge Resistors
                         //CLKOut=1Hz
                         //Enable Thermometer
                         //Measure each 16 seconds
  Wire.endTransmission(); 

  Serial.begin(9600);
  delay(10000); 
  if (!sd.begin(chipSelect, SPI_HALF_SPEED)) sd.initErrorHalt(); // SPI_HALF_SPEED to avoid bus errors, SPI_FULL_SPEED for better performance 
  SetupFilename();
  CSVSetPTR = &CSVSet;
  if (!file.open(fileName, O_CREAT | O_WRITE | O_EXCL)) error("file.open");
  do {
    delay(10);
  } 
  while (Serial.read() >= 0);
  Serial.println("Jetilog Rev3.0");
  Serial.print(F("Logging to: "));
  Serial.println(fileName);

  SetupMSG();  
  EnableIRQRx1();
  EnableIRQAnalogComp();  
  sei();
  CSVSetPTR = &CSVSet;
  TSLog = millis()/Samplingrate_ms + 1;//calculate integer multiple of Samplingrate_ms +1
  TSLog = TSLog*Samplingrate_ms; //calculate corresponding timestamp in us
  if (filledPTR->msgstatus==2) {
     oldPTR=filledPTR;
     oldPTR->msgstatus=3;
      //filledPTR zeigt auf MSG(0)     
  }
}

void loop() {
   String data; //User Input String
  if (SDCloseFile)PowerDown();//close File if powerdown-request occured
  //--------------------------RTC
  if (Serial.available() > 0)
  {
    data=Serial.readStringUntil('\n');
  
    if(!data.compareTo("?")) rv3029_print();  
    if (data.length()==13) //Date/Time-Command: date tt.mm.yy time hh:mm:ss
      {
      if (!data.substring(0,4).compareTo("date")) 
        { 
        Wire.beginTransmission(RV3029Write);
        Wire.write(ClockADR+3);
        Wire.write(str2bcd(data[5],data[6]));
        Wire.write(0x00);
        Wire.write(str2bcd(data[8],data[9]));
        Wire.write(str2bcd(data[11],data[12]));
        Wire.endTransmission();
        }
      else 
        {
        if (!data.substring(0,4).compareTo("time"))      
          { 
          Wire.beginTransmission(RV3029Write);
          Wire.write(ClockADR);
          Wire.write(str2bcd(data[11],data[12]));
          Wire.write(str2bcd(data[8],data[9]));
          Wire.write(str2bcd(data[5],data[6]) & B10111111);//24h-mode
          Wire.endTransmission();
          }
        }
      rv3029_print();
    }
  }
  //--------------------------RTC
  
  if (filledPTR->msgstatus==2) { //a new Message has received 
    debug=HasDifferences(filledPTR,oldPTR);
    if ((LogCtrl&(1<<LnC))||(HasDifferences(filledPTR,oldPTR)!=0)){
      cli();
      oldPTR->msgstatus=0;//0=empty, 1=RxReserved, 2=filled, 3=LogReserved
      emptyPTR=oldPTR;
      filledPTR->msgstatus=3;//0=empty, 1=RxReserved, 2=filled, 3=LogReserved
      Msg2CSV(filledPTR,CSVSetPTR);
      oldPTR=filledPTR;
      sei();
      if ((LogCtrl&(1<<LnT))||(TSLog<millis())) {//if msg-content has changed and Time elapsed, dump content to SD-Card
        TSLog = millis()+Samplingrate_ms;//next interesting Timestamp for trigger Log
        
        if ((LogCtrl&(1<<LaM))||(CSVSet.hasNumbers)){
          if (msgcounter==0) TSStart=millis();
          msgcounter++;
          Serial.print(msgcounter);
          Serial.print(";");
          Serial.print((millis()-TSStart)/1000ul);
          Serial.print(";");
          Serial.print(CSVSet.text);
          Serial.print("/");
          Serial.println(debug);
      
          file.print(msgcounter);
          file.write(';');
          file.print((millis()-TSStart)/1000ul);
          file.write(';');
          file.println(CSVSet.text);

          if (SDCloseFile)PowerDown();//close File if powerdown-request occured     
          if (!file.sync() || file.getWriteError()) error("write error"); // Force data to SD and update the directory entry to avoid data loss.     
        }
     }//end TSLog<millis()
     else{
     }
    }
    else{ 
      cli();
      filledPTR->msgstatus=0;//free message
      emptyPTR=filledPTR; 
      sei();
    }//end Has Difference   
   }//end new message
   idleCnt++;
}

void rv3029_print() 
{
  byte data[8];
  byte index=0;
  
  Wire.beginTransmission(RV3029Write);
  Wire.write(ClockADR);
  Wire.endTransmission();
  
  Wire.requestFrom(RV3029Write, 8);  
  while(Wire.available()) 
  { 
    data[index] = Wire.read(); 
    index++;
  }
  printhex(data[3]);     
  Serial.print(".");  
  printhex(data[5]);
  Serial.print(".");  
  printhex(data[6]);
  Serial.print(" ");  
  printhex(data[2] & B00111111);
  Serial.print(":");  
  printhex(data[1]);
  Serial.print(":");  
  printhex(data[0]);
  Serial.println();
}

byte str2bcd(char data1,char data2) 
{
  if (data1>'9') return STR2BCD_ERROR;
  if (data1<'0') return STR2BCD_ERROR;
  if (data2>'9') return STR2BCD_ERROR;
  if (data2<'0') return STR2BCD_ERROR;

  return ((data1-0x30)<<4)+(data2-0x30);
}


void printhex(byte data) 
{
  char buffer[4];
  sprintf(buffer,"%02x",data);
  Serial.print(buffer);
}
